/*                                                                            */
/*      Sample OS/2 PM Queue Processor                                        */
/*      Author: Sam Detweiler                                                 */
/*                                                                            */
/*      As documented in the Presentation Driver Interface Manual, a Queue    */
/*      Processor is responsible for printing a spool file created by an      */
/*      Application using DevOpenDC("OD_QUEUED").                             */
/*                                                                            */
/*      To this end a Queue processor Supplies the following callable         */
/*      entrypoints.                                                          */
/*                                                                            */
/*      SplQpInstall(hwnd)      - Only called once, at Control Panel          */
/*                                install time. hwnd, is the window           */
/*                                handle of the Control Panel, to be used     */
/*                                if a Dialog is required with the user.      */
/*                                                                            */
/*                              Returns - TRUE   - installed ok               */
/*                                        FALSE  - install failed             */
/*                                                                            */
/*                                See Note below on SplQpQueryDt              */
/*                                                                            */
/*      SplQpQueryDt(pcount,pdata) -Called at any time to determine what      */
/*                                 data types this queue processor supports   */
/*                                                                            */
/*                                 MUST support PM_Q_STD and PM_Q_RAW         */
/*                                 at minimum.                                */
/*                                                                            */
/*                                 Normal call sequence is twice, once        */
/*                                 with pcount=0 to get the count of types    */
/*                                      pdata should NOT be modified          */
/*                                 with pcount!=0 to get that many            */
/*                                                                            */
/*******Note that SplQpQueryDt MUST be EXPORTED for the Control Panel**********/
/*******install pull down to find this processor.                    **********/
/*                                                                            */
/*      The following entrypoints are listed in the order they are usually    */
/*      called.                                                               */
/*                                                                            */
/*                                                                            */
/*      SplQpOpen    - Open Queue Processor  - Once per spool file            */
/*                                                                            */
/*                     Processing of this call is as follows                  */
/*                                                                            */
/*                     DosAllocSeg     segment for heap                       */
/*                     WinCreateHeap   Create heap from supplied selector     */
/*                     WinAllocMem     get memory for QPOPENSTRUC             */
/*                        Copy Parm(s) in loop                                */
/*                           if(parm)                                         */
/*                             {                                              */
/*                             WinAllocMem(strlen(parm)+1)                    */
/*                             CopyParm                                       */
/*                             }                                              */
/*                     hproc->aborting=No;                                    */
/*                     hproc->semiphore=Null;                                 */
/*                     hproc->state=Running;                                  */
/*                     return Pointer to QPOPENSTRUC as HPROC                 */
/*                                                                            */
/*      SplQpPrint   - Print Spool file      - Once per spool file            */
/*                                                                            */
/*                     Processing of this call is as follows (page 17-20)     */
/*                                                                            */
/*                     DosGetPid(hproc->threadid);                            */
/*                     DevOpenDC (OD_DIRECT, datatype);                       */
/*                                                                            */
/*                        switch (datatype)                                   */
/*                                                                            */
/*                           Case PM_Q_STD:                                   */
/*                                                                            */
/*                                GpiCreatePS(hps,Normal);                    */
/*                                DevEscape (DEVESC_STARTDOC);                */
/*                                GpiAssociate(hps,hdc);                      */
/*                                GpiLoadMetaFile(filename,hmf);              */
/*                                if(GpiPlayMetaFile(hps,hmf)!=GPI_OK)        */
/*                                  {                                         */
/*                                  if(WinGetLastError==PMERR_STOP_DRAW)      */
/*                                    {                                       */
/*                                    DevEscape(DEVESC_ABORTDOC);             */
/*                                    DosSemClear(hproc->semiphore);          */
/*                                    }                                       */
/*                                  }                                         */
/*                                GpiDeleteMetaFile(hmf);                     */
/*                                GpiAssociate(hps,NULL);                     */
/*                                GpiDestroyPS();                             */
/*                                break;                                      */
/*                                                                            */
/*                           Case PM_Q_RAW:                                   */
/*                                                                            */
/*                                DevEscape (DEVESC_STARTDOC);                */
/*                                DosOpen(spoolfile);                         */
/*                                   Do til eof(spoolfile)                    */
/*                                     if(Aborting==No)                       */
/*                                       {                                    */
/*                                       DosSemRequest(hproc->semiphore);     */
/*                                       DosRead(hproc->handle);              */
/*                                       DevEscape(DEVESC_RAWDATA);           */
/*                                       DosSemClear(hproc->semiphore);       */
/*                                       }                                    */
/*                                     else                                   */
/*                                       break;                               */
/*                                   end                                      */
/*                                if(hproc->state==Running)                   */
/*                                  DosClose(spoolfile);                      */
/*                                break;                                      */
/*                                                                            */
/*                           Case Private_datatype:                           */
/*                                                                            */
/*                                as required                                 */
/*                                break;                                      */
/*                                                                            */
/*                     DevEscape (DEVESC_ENDDOC);                             */
/*                                                                            */
/*                     DevCloseDC();                                          */
/*                                                                            */
/*      SplQpClose      - Close Queue Processor - Once per spool file         */
/*                      WinDestroyHeap(hproc->heaphandle);                    */
/*                      DosFreeSeg(segment);                                  */
/*                                                                            */
/*      SplQpControl    - Control printing of current Spool file              */
/*                                                                            */
/*                                                                            */
/*                        - SPLC_ABORT  - abort print out now                 */
/*                              if(datatype==PM_Q_STD)                        */
/*                                {                                           */
/*                                DosSemSet(hproc->semiphore)                 */
/*                                GpiSetStopDraw(hproc->hps,SDW_ON);          */
/*                                if(hproc->state==Paused)                    */
/*                                  DosResumeThread(hproc->threadid)          */
/*                                DosSemWait(hproc->semiphore)                */
/*                                GpiSetStopDraw(hproc->hps,SDW_OFF);         */
/*                                }                                           */
/*                              else                                          */
/*                                {                                           */
/*                                if(hproc->state==Running)                   */
/*                                  DosSemRequest(hproc->semiphore)           */
/*                                hproc->Aborting=Yes                         */
/*                                DosSemClear(hproc->semiphore)               */
/*                                }                                           */
/*                        - SPLC_PAUSE  - Pause printout now                  */
/*                              if(hproc->state==Running)                     */
/*                                {                                           */
/*                                if(datatype==PM_Q_STD)                      */
/*                                  DosSuspendThread(hproc->threadid)         */
/*                                else                                        */
/*                                  {                                         */
/*                                  DosSemRequest(hproc->semiphore)           */
/*                                  DosChgFilePtr(handle,0,1,hproc->lastread) */
/*                                  DosClose(hproc->handle);                  */
/*                                  }                                         */
/*                                hproc->state=Paused;                        */
/*                                }                                           */
/*                        - SPLC_CONTINUE - Resume paused printout now        */
/*                              if(hproc->state==Paused)                      */
/*                                {                                           */
/*                                hproc->state=Running;                       */
/*                                if(datatype==PM_Q_STD)                      */
/*                                  DosResumeThread(hproc->threadid)          */
/*                                else                                        */
/*                                  {                                         */
/*                                  DosOpen(hproc->spoolfile);                */
/*                                  DosChgFilePtr(handle,hproc->lastread,0,x) */
/*                                  DosSemClear(hproc->semiphore)             */
/*                                  }                                         */
/*                                }                                           */
/*      The Spooler Queue Manager can ask the Queue Processor to              */
/*      change its processing of the currently active spool file with         */
/*      this call. Note that this call comes in on a separate                 */
/*      thread, and so is asyncronous. The main thread (processing the        */
/*      SplQpPrint call) must periodically examine a shared flag and          */
/*      act accordingly.                                                      */
/*                                                                            */
/*        Note also, that if SplQpPrint is processing a PM_Q_STD file         */
/*        it has made a GpiPlayMetaFile call, and will not get control        */
/*        back til its done. This call should SUSPEND the SplQpPrint          */
/*        thread to handle the SPLC_SUSPEND request.                          */
/*                                                                            */
/*      Change History:                                                       */
/*                                                                            */
/*        Created: 11/18/89                                                   */
/*                 11/27/89     - Add info on SplQpControl Processing         */
/*                 12/04/89     - Get running for PM_Q_RAW Processing         */
/*                                                                            */

#define MINHEAP 2000
#define NonShared 0
#define Zero    0
#define ZeroL   0L

#define Running 0
#define Paused  1

#define No      0
#define Yes     1

#define Raw     1
#define RawString       "PM_Q_RAW"
#define Std     2
#define StdString       "PM_Q_STD"
#define Private 3
#define PvtString       "PVT_Q_DATA"
#define NumTypes 3
#define TypeSize 16
#define BufSize 2048

#define SPLC_ABORT 0L
#define SPLC_PAUSE 1L
#define SPLC_CONTINUE 2L

#define INCL_SPL
#define INCL_WIN
#define INCL_DEV
#define INCL_DOSMEMMGR
#define INCL_DOSSEMAPHORES
#define INCL_DOSFILEMGR
#define INCL_DOSPROCESS
#define INCL_GPICONTROL
#define INCL_GPIPRIMITIVES
#define INCL_GPIMETAFILES
#define INCL_GPIERRORS
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>

CHAR datatypes[NumTypes][TypeSize]={{RawString},{StdString},{PvtString}};
PSZ CopyString(HHEAP heapHandle,SEL Sel,PSZ From);
PSZ CopyData(HHEAP heapHandle,SEL Sel,PBYTE From,USHORT len);

typedef struct _QPOPENSTRUC { /* dop */
    PSZ        pszLogAddress;
    PSZ        pszDriverName;
    PDRIVDATA  pdriv;
    PSZ        pszDataType;
    PSZ        pszComment;
    PSZ        pszQueueProcParams;
    PSZ        pszSpoolerParams;
    PSZ        pszNetworkParams;
    PSZ        pszDocName;
    PSZ        pszQueueProcName;
    PSZ        pszToken;
    USHORT     JobId;
} QPOPENSTRUC;
typedef QPOPENSTRUC FAR *PQPOPENSTRUC;
#define QPOPENSIZE sizeof(QPOPENSTRUC)

typedef struct _QPSTRUC {
        ULONG count;
        QPOPENSTRUC qp;
        PSZ    spoolfile;
        HHEAP  hhandle;
        BYTE   state;
        BYTE   Aborting;
        ULONG  lastread;
        HFILE  handle;
        HDC    hdc;
        HPS    hps;
        BYTE   datatype;
        HSEM   semiphore;
        PIDINFO pidinfo;
        BYTE   readbuf[BufSize];
} QPSTRUC;
typedef QPSTRUC FAR *PQPSTRUC;
#define QPOPSIZE sizeof(QPSTRUC)

BOOL EXPENTRY SplQpInstall(HWND hwnd)
{
BOOL flag;
        return TRUE;
}

HPROC EXPENTRY SplQpOpen(LONG cbData, PQPOPENSTRUC opendata)
{
PQPSTRUC qphandle;
HHEAP heapHandle;
SEL selector;
USHORT offset,len,i;
PBYTE cp,savecp;
#ifdef Debug
FILE *f;
#endif
        DosAllocSeg(QPOPSIZE,&selector,NonShared);
        heapHandle=WinCreateHeap(selector,0,0,0,0,0);
        offset=(USHORT)WinAllocMem(heapHandle,QPOPSIZE);
        qphandle=MAKEP(selector,offset);
        memset(qphandle,Zero,QPOPSIZE);
        memcpy((char *)&qphandle->qp,(char *)opendata,QPOPENSIZE);
        qphandle->count=cbData;
        qphandle->hhandle=heapHandle;
        qphandle->qp.pszLogAddress=CopyString(heapHandle, selector,
                                         opendata->pszLogAddress);
        qphandle->qp.pszDriverName=CopyString(heapHandle, selector,
                                         opendata->pszDriverName);
        qphandle->qp.pdriv=(PDRIVDATA)CopyData(heapHandle, selector,
                                         (PBYTE)opendata->pdriv,(USHORT)opendata->pdriv->cb);
        qphandle->qp.pszDataType=CopyString(heapHandle, selector,
                                         opendata->pszDataType);
        if(!strcmpi(opendata->pszDataType,RawString))
          qphandle->datatype=Raw;
        else if(!strcmpi(opendata->pszDataType,StdString))
          qphandle->datatype=Std;
        else if(!strcmpi(opendata->pszDataType,PvtString))
          qphandle->datatype=Private;
        qphandle->qp.pszComment=CopyString(heapHandle, selector,
                                         opendata->pszComment);
        qphandle->qp.pszQueueProcName=CopyString(heapHandle, selector,
                                         opendata->pszQueueProcName);
        qphandle->qp.pszQueueProcParams=CopyString(heapHandle, selector,
                                         opendata->pszQueueProcParams);
        qphandle->qp.pszSpoolerParams=CopyString(heapHandle, selector,
                                         opendata->pszSpoolerParams);
        qphandle->qp.pszNetworkParams=CopyString(heapHandle, selector,
                                         opendata->pszNetworkParams);
        qphandle->qp.pszDocName=CopyString(heapHandle, selector,
                                         opendata->pszDocName);
        qphandle->qp.pszToken=CopyString(heapHandle, selector,
                                         opendata->pszToken);
        cp=NULL;
        WinQueryProfileSize((HAB)NULL,"PM_SPOOLER_QUEUE",
             qphandle->qp.pszQueueProcName,&len);
        DosAllocSeg(len+1,(PSEL)&FP_SEG(cp),0);
        WinQueryProfileString((HAB)NULL,"PM_SPOOLER_QUEUE",
             qphandle->qp.pszQueueProcName,";;;",cp,len+1);
        savecp=cp;
        /*                                              */
        for(i=0;i<2 && cp;i++,cp++)
           cp=strchr(cp,';');
        /*                                              */
        /*      cp now points to Network parms IF ANY   */
        /*                                              */
        /*      make sure to save them for SplQpPrint   */
        /*                                              */
        /*      'MAY' be able to dialog with user here  */
        /*      if they are missing. See WinDlgBox      */
        /*                                              */
        DosFreeSeg(FP_SEG(savecp)); /* don't forget to free segment */

#ifdef Debug
        f=fopen("c:\\pmqp.trc","w");
        fprintf(f,"Logical Address = %s\n", qphandle->qp.pszLogAddress);
        fprintf(f,"Driver Name = %s\n", qphandle->qp.pszDriverName);
        fprintf(f,"Data Type = %s\n", qphandle->qp.pszDataType);
        fprintf(f,"Comment = %s\n", qphandle->qp.pszComment);
        fprintf(f,"Queue Processor Name = %s\n", qphandle->qp.pszQueueProcName);
        fprintf(f,"Queue Processor Parms = %s\n", qphandle->qp.pszQueueProcParams);
        fprintf(f,"Spooler Parms = %s\n", qphandle->qp.pszSpoolerParams);
        fprintf(f,"Network Parms = %s\n", qphandle->qp.pszNetworkParams);
        fprintf(f,"Document Name= %s\n", qphandle->qp.pszDocName);
        fprintf(f,"Token = %s\n", qphandle->qp.pszToken);
        fprintf(f,"Job ID = %d\n", qphandle->qp.JobId);
        fclose(f);
#endif
        return (HPROC)qphandle;
}

BOOL EXPENTRY SplQpPrint(PQPSTRUC qphandle,PSZ spoolfile)
{
BOOL flag;
HMF hmf;
USHORT returned;
ULONG readsize;
BYTE desc[254];
LONG SegCount;
        qphandle->spoolfile=CopyString(qphandle->hhandle, SELECTOROF(qphandle),
                                         spoolfile);
        DosGetPID(&qphandle->pidinfo);
        qphandle->hdc=DevOpenDC((HAB)NULL,OD_DIRECT,"*",4L,
                                (PDEVOPENDATA)&qphandle->qp,NULL);
        switch (qphandle->datatype)
          {
          case Std:

            qphandle->hps=GpiCreatePS((HAB)NULL,qphandle->hdc,ZeroL,
                 PU_ARBITRARY | GPIF_DEFAULT | GPIT_NORMAL | GPIA_ASSOC);
            DevEscape(qphandle->hdc,DEVESC_STARTDOC, ZeroL, NULL,ZeroL,NULL);
            hmf=GpiLoadMetaFile((HAB)NULL,spoolfile);
            if(GpiPlayMetaFile(qphandle->hps,hmf,ZeroL,NULL,&SegCount,
                               (LONG)sizeof(desc),desc)==GPI_ERROR)
              {
              if(WinGetLastError((HAB)NULL)==PMERR_STOP_DRAW_OCCURRED)
                {
                DevEscape(qphandle->hdc,DEVESC_ABORTDOC,ZeroL,NULL,ZeroL,NULL);
                DosSemClear(&qphandle->semiphore);
                }
              }
            GpiDeleteMetaFile(hmf);
            GpiAssociate(qphandle->hps,NULL);
            GpiDestroyPS(qphandle->hps);
            break;

          case Raw:

            DevEscape(qphandle->hdc,DEVESC_STARTDOC, ZeroL, NULL,ZeroL,NULL);
            DosOpen(qphandle->spoolfile,&qphandle->handle,&returned,
                    NULL,FILE_NORMAL,OPEN_ACTION_OPEN_IF_EXISTS,
                    OPEN_ACCESS_READONLY|OPEN_SHARE_DENYNONE ,NULL);
            for(readsize=BufSize;readsize!=Zero;)
              {
              DosSemRequest(&qphandle->semiphore,SEM_INDEFINITE_WAIT);
              if(qphandle->Aborting==No)
                {
                DosRead(qphandle->handle,qphandle->readbuf,BufSize,(PUSHORT)&readsize);
                if(readsize!=Zero)
                  DevEscape(qphandle->hdc,DEVESC_RAWDATA,
                            readsize,(PBYTE)qphandle->readbuf,ZeroL,NULL);
                DosSemClear(&qphandle->semiphore);
                }
              else
                {
                DosSemClear(&qphandle->semiphore);
                break;
                }
              }
            if(qphandle->state==Running)
              DosClose(qphandle->handle);
            break;

          default:
            break;
          }

        DevEscape(qphandle->hdc,DEVESC_ENDDOC,ZeroL,NULL,ZeroL,NULL);

        DevCloseDC(qphandle->hdc);
        return TRUE; /* don't know under what conditions this should be false */
}

BOOL EXPENTRY SplQpClose(PQPSTRUC qphandle)
{
        WinDestroyHeap(qphandle->hhandle);
        DosFreeSeg(SELECTOROF(qphandle));
        return TRUE;
}

BOOL EXPENTRY SplQpQueryDt(PLONG count,PSZ *data)
{
BOOL flag;
LONG work;
USHORT i;
        if(*count)
          {
          work=min(*count,NumTypes);
          *count=work;
          for(i=0;i<(USHORT)work;i++)
            strcpy(data[i],datatypes[i]);
          }
        else
          *count=NumTypes;
        return TRUE;
}

BOOL EXPENTRY SplQpControl(PQPSTRUC qphandle,LONG Code)
{
BOOL flag;
ULONG x;
USHORT returned;
        flag = TRUE;
        switch((USHORT)Code)
          {
          case SPLC_ABORT:  /* abort print out now */
              if(qphandle->datatype==Std)
                {
                DosSemSet(&qphandle->semiphore);
                GpiSetStopDraw(qphandle->hps,SDW_ON);
                if(qphandle->state==Paused)
                  DosResumeThread(qphandle->pidinfo.tid);
                DosSemWait(&qphandle->semiphore,SEM_INDEFINITE_WAIT);

                /* Not required, as HPS is being destroyed   */
                /* GpiSetStopDraw(qphandle->hps,SDW_OFF);    */
                /*                                           */
                }
              else
                {
                if(qphandle->state==Running)
                  DosSemRequest(&qphandle->semiphore,SEM_INDEFINITE_WAIT);
                qphandle->Aborting=Yes;
                DosSemClear(&qphandle->semiphore);
                }
              break;
          case SPLC_PAUSE:  /* Pause printout now */
              if(qphandle->state==Running)
                {
                if(qphandle->datatype==Std)
                  DosSuspendThread(qphandle->pidinfo.tid);
                else
                  {
                  DosSemRequest(&qphandle->semiphore,SEM_INDEFINITE_WAIT);
                  DosChgFilePtr(qphandle->handle,ZeroL,FILE_CURRENT,
                         &qphandle->lastread);
                  DosClose(qphandle->handle);
                  }
                qphandle->state=Paused;
                }
              break;
          case SPLC_CONTINUE: /*  Resume paused printout now */
              if(qphandle->state==Paused)
                {
                qphandle->state=Running;
                if(qphandle->datatype==Std)
                  DosResumeThread(qphandle->pidinfo.tid);
                else
                  {
                  DosOpen(qphandle->spoolfile,&qphandle->handle,&returned,
                          NULL,FILE_NORMAL,OPEN_ACTION_OPEN_IF_EXISTS,
                          OPEN_ACCESS_READONLY,NULL);
                  DosChgFilePtr(qphandle->handle,qphandle->lastread,
                          FILE_BEGIN,&x);
                  DosSemClear(&qphandle->semiphore);
                  }
                }
              break;
          default:
              break;
          }
        return flag;
}
PSZ CopyString(HHEAP heapHandle,SEL Sel,PSZ From)
{
USHORT offset,len;
PSZ temp;
        temp=NULL;
        if(From)
          len=strlen(From);
        else
          len=Zero;
        if(len && (offset=(USHORT)WinAllocMem(heapHandle,len+1)))
          {
          temp=MAKEP(Sel,offset);
          strcpy(temp,From);
          }
        return temp;
}
PSZ CopyData(HHEAP heapHandle,SEL Sel,PBYTE From,USHORT len)
{
USHORT offset;
PSZ temp;
        temp=NULL;
        if(len && (offset=(USHORT)WinAllocMem(heapHandle,len)))
          {
          temp=MAKEP(Sel,offset);
          memcpy(temp,From,len);
          }
        return temp;
}
